/* ==================================================================
File        : blurs.hlsl
Author      : Various
Date        : 2007
Format      : HLSL/Cg
Description : Blur operators

================================================================== */

#include "default.hlsl"

//------------------------------------------------------------------------------------------------
// Texture samplers
//------------------------------------------------------------------------------------------------

#define diffuse_tex layer0_sampler

//@:-texformat normalGlow_tex RGBA8
sampler2D normalGlow_tex      : register(s1);
sampler2D depth_tex			  : register(s2);
sampler2D CoC_tex			  : register(s3);	// CoC precomputed values for HQ DOF blur

//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------

float4 tint					  : register(c30);
float4 offsetWeights[13]      : register(c160);
float4 scaleBias              : register(c173); // { scale_x, scale_y, bias_x, bias_y }

//------------------------------------------------------------------------------------------------
// Vertex output structs
//------------------------------------------------------------------------------------------------

struct VS_OUT5X5
{
	half4 position : POSITION;
	half4 offsets [7] : TEXCOORD0; // xy, then zw contains offsets, weights is always 1 for all 13 samples
};

struct VS_OUT7X7
{
	float4 position : POSITION;
	half4 offsetWeights [7] : TEXCOORD0;
};

struct VS_OUT1
{
	float4 position : POSITION;
	half4 offsetWeights [1] : TEXCOORD0;
};

struct VS_OUT2
{
	float4 position : POSITION;
	half4 offsetWeights [2] : TEXCOORD0;
};

struct VS_OUT3
{
	float4 position : POSITION;
	half4 offsetWeights [3] : TEXCOORD0;
};

struct VS_OUT4
{
	float4 position : POSITION;
	half4 offsetWeights [4] : TEXCOORD0;
};

struct VS_OUT5
{
	float4 position : POSITION;
	half4 offsetWeights [5] : TEXCOORD0;
};

struct VS_OUT7
{
	float4 position : POSITION;
	half4 offsetWeights [7] : TEXCOORD0;
};

struct VS_OUT9
{
	float4 position : POSITION;
	half4 offsetWeights [9] : TEXCOORD0;
};


//------------------------------------------------------------------------------------------------
// blur5x5
//------------------------------------------------------------------------------------------------

/**
 * Vertex shader for the blur5x5 type pixel shaders.
 */
VS_OUT5X5 blur5x5_vx( VS_IN_2D input)
{
	VS_OUT5X5 vout;
	
	vout.position = input.position;

#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif
	
	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	// scale and bias texture coordinates
	for (int i = 0; i < 6; i++)
	{
		vout.offsets[i].xy = texcoord + offsetWeights[i*2].xy;
		vout.offsets[i].xy = vout.offsets[i].xy * scaleBias.xy + scaleBias.zw;
		
		vout.offsets[i].zw = texcoord + offsetWeights[i*2 + 1].xy;
		vout.offsets[i].zw = vout.offsets[i].zw * scaleBias.xy + scaleBias.zw;
	}
	
	vout.offsets[6].xy = texcoord + offsetWeights[12].xy;
	vout.offsets[6].xy = vout.offsets[6].xy * scaleBias.xy + scaleBias.zw;

	// not set
	vout.offsets[6].zw = half2(0.0, 0.0);
	
	return vout;
}

//_________________________________________________________________________________________________

/**
 * Diamond filtering (13 samples).
  <pre>
    *
   ***
  *****
   ***
    *
  </pre>
 */
half4 blur5x5_px( VS_OUT5X5 In ) : COLOR
{
    half4 vColor = 0.0;

    for( int i=0; i < 6; i++ )
    {
	    half4 c;
	    
	    c = tex2D(diffuse_tex, In.offsets[i].xy);
        vColor += c;
        
	    c = tex2D(diffuse_tex, In.offsets[i].zw);
        vColor += c; // can optimize
    }
    
	half4 c = tex2D(diffuse_tex, In.offsets[6].xy);
	vColor += c;

	vColor /= 13.0;

    return vColor;
}

//------------------------------------------------------------------------------------------------
// blur7x7
//------------------------------------------------------------------------------------------------

/**
 * Vertex shader for the blur7x7 type pixel shaders.
 */
VS_OUT7X7 blur7x7_vx( VS_IN_2D input)
{
	VS_OUT7X7 vout;
	
	vout.position = input.position;

#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif
	
	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 7; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;

		// scale and bias texture coordinates
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
		
	return vout;
}

//_________________________________________________________________________________________________

/**
 * Two pass filtering (7 samples per pass).
 */
half4 blur7x7_px( VS_OUT7X7 In ) : COLOR
{
	half4 vColor = In.offsetWeights[0].z * tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += In.offsetWeights[1].z * (tex2D(diffuse_tex, In.offsetWeights[1].xy) + tex2D(diffuse_tex, In.offsetWeights[4].xy));
	vColor += In.offsetWeights[2].z * (tex2D(diffuse_tex, In.offsetWeights[2].xy) + tex2D(diffuse_tex, In.offsetWeights[5].xy));
	vColor += In.offsetWeights[3].z * (tex2D(diffuse_tex, In.offsetWeights[3].xy) + tex2D(diffuse_tex, In.offsetWeights[6].xy));
    return vColor;
}

//_________________________________________________________________________________________________

/**
 * Two pass filtering (7 samples per pass), but with only alpha component blurred. The other
 * components (rgb) take the value of the first sample (with weight 1).
 */
half4 blur7x7alpha_px( VS_OUT7X7 In ) : COLOR
{
	half4 vColor = half4(1.0, 1.0, 1.0, In.offsetWeights[0].z) * tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor.a += In.offsetWeights[1].z * (tex2D(diffuse_tex, In.offsetWeights[1].xy).a + tex2D(diffuse_tex, In.offsetWeights[4].xy).a);
	vColor.a += In.offsetWeights[2].z * (tex2D(diffuse_tex, In.offsetWeights[2].xy).a + tex2D(diffuse_tex, In.offsetWeights[5].xy).a);
	vColor.a += In.offsetWeights[3].z * (tex2D(diffuse_tex, In.offsetWeights[3].xy).a + tex2D(diffuse_tex, In.offsetWeights[6].xy).a);
    return vColor;
}

//_________________________________________________________________________________________________


inline float unpackRGBDepth(float3 rgbDepth)
{
	return dot(round(rgbDepth * 255.0), float3(65536.0/16777215.0, 256.0/16777215.0, 1.0/16777215.0));
}

//_________________________________________________________________________________________________


// Two pass filtering (7 samples per pass).
// The RGB channels contain the depth information and the alpha channel contains the lod.

float4 blur7x7dof_px( VS_OUT7X7 In ) : COLOR
{
	float4 s0 = tex2D(diffuse_tex, In.offsetWeights[0].xy);
	float4 s1 = tex2D(diffuse_tex, In.offsetWeights[1].xy);
	float4 s2 = tex2D(diffuse_tex, In.offsetWeights[2].xy);
	float4 s3 = tex2D(diffuse_tex, In.offsetWeights[3].xy);
	float4 s4 = tex2D(diffuse_tex, In.offsetWeights[4].xy);
	float4 s5 = tex2D(diffuse_tex, In.offsetWeights[5].xy);
	float4 s6 = tex2D(diffuse_tex, In.offsetWeights[6].xy);

	// Unpack the RGB depth values into the blue channel.
	s0.b = unpackRGBDepth(s0.rgb);
	s1.b = unpackRGBDepth(s1.rgb);
	s2.b = unpackRGBDepth(s2.rgb);
	s3.b = unpackRGBDepth(s3.rgb);
	s4.b = unpackRGBDepth(s4.rgb);
	s5.b = unpackRGBDepth(s5.rgb);
	s6.b = unpackRGBDepth(s6.rgb);

	// Compute the weighted average of both the unpacked depth and the LOD
	float2 avg;
	avg = s0.ba * In.offsetWeights[0].z + (s1.ba + s4.ba) * In.offsetWeights[1].z + (s2.ba + s5.ba) * In.offsetWeights[2].z + (s3.ba + s6.ba) * In.offsetWeights[3].z;

	// Return the packed average depth and LOD
	float3 d;
	float iDepth = 256.0 * modf(avg.x * 16777215.0/65536.0, d.x);
	d.z = modf(iDepth, d.y);
	d.xy /= 255.0;
	return float4(d, avg.y);
}
//_________________________________________________________________________________________________


// Two pass filtering (5 samples per pass).
// The RGB channels contain the depth information and the alpha channel contains the lod.

float4 blur5x5dof_px( VS_OUT5 In ) : COLOR
{
	float4 s0 = tex2D(diffuse_tex, In.offsetWeights[0].xy);
	float4 s1 = tex2D(diffuse_tex, In.offsetWeights[1].xy);
	float4 s2 = tex2D(diffuse_tex, In.offsetWeights[2].xy);
	float4 s3 = tex2D(diffuse_tex, In.offsetWeights[3].xy);
	float4 s4 = tex2D(diffuse_tex, In.offsetWeights[4].xy);

	// Unpack the RGB depth values into the blue channel.
	s0.b = unpackRGBDepth(s0.rgb);
	s1.b = unpackRGBDepth(s1.rgb);
	s2.b = unpackRGBDepth(s2.rgb);
	s3.b = unpackRGBDepth(s3.rgb);
	s4.b = unpackRGBDepth(s4.rgb);

	// Compute the weighted average of both the unpacked depth and the LOD
	float2 avg;
	avg = s0.ba * In.offsetWeights[0].z + (s1.ba + s3.ba) * In.offsetWeights[1].z + (s2.ba + s4.ba) * In.offsetWeights[2].z;

	// Return the packed average depth and LOD
	float3 d;
	float iDepth = 256.0 * modf(avg.x * 16777215.0/65536.0, d.x);
	d.z = modf(iDepth, d.y);
	d.xy /= 255.0;
	return float4(d, avg.y);
}

//------------------------------------------------------------------------------------------------
// Shaders for 1-5 tap blurs (usable for seperable blurs up to 10x10)
//------------------------------------------------------------------------------------------------

float4 blur2hdr_px( VS_OUT2 In ) : COLOR
{
	float4 vColor = 0.5 * (tex2D(diffuse_tex, In.offsetWeights[0].xy) + tex2D(diffuse_tex, In.offsetWeights[1].xy));
	float intensity = max(max(vColor.r, vColor.g), vColor.b);

	// Hard-code a pow 5 function...
	float bloomIntensity = intensity * intensity;
	bloomIntensity = bloomIntensity * bloomIntensity;
	vColor *= bloomIntensity * intensity;
	
    return float4(vColor.rgb * tint.rgb, 1);
}

//_________________________________________________________________________________________________

float4 blur4hdr_px( VS_OUT4 In ) : COLOR
{
	// HDR tone curve takes around 7 instructions, we can either apply it on the source
	// image, or we can downsample the image first and then apply the curve

	// Apply HDR curve to every sample individually (slower)
    /*for( int i=0; i < 4; i++ )
    {
		half4 c = tex2D(diffuse_tex, In.offsetWeights[i].xy);
		//float intensity = max( max ( c.r, c.g ), c.b );
		float intensity = dot( c.rgb, float3(0.33,0.33,0.33) );
		float bloomIntensity = pow( intensity, toneCurvePow );
		c.rgb = c.rgb * bloomIntensity;
		vColor += In.offsetWeights[i].z * c;
	}*/
	
	// Blur image first, then apply HDR curve to result (faster)
	float4 vColor = tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += tex2D(diffuse_tex, In.offsetWeights[1].xy);
	vColor += tex2D(diffuse_tex, In.offsetWeights[2].xy);
	vColor += tex2D(diffuse_tex, In.offsetWeights[3].xy);
	vColor *= 0.25;

	float intensity = max( max ( vColor.r, vColor.g ), vColor.b );
	//float intensity = dot( vColor.rgb, float3(0.33,0.33,0.33) )

//	float toneCurvePow = 5.0;
//	float bloomIntensity = pow( intensity, toneCurvePow );
//	vColor *= bloomIntensity;

	// Hard-code a pow 5 function...
	float bloomIntensity = intensity * intensity;
	bloomIntensity = bloomIntensity * bloomIntensity;
	vColor *= bloomIntensity * intensity;
	
    return float4(vColor.rgb * tint.rgb, 1);
}

//_________________________________________________________________________________________________

VS_OUT1 blur1_vx( VS_IN_2D input)
{
	VS_OUT1 vout;
	vout.position = input.position;
#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif

	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	vout.offsetWeights[0] = offsetWeights[0];
	vout.offsetWeights[0].xy += texcoord;
	vout.offsetWeights[0].xy = vout.offsetWeights[0].xy * scaleBias.xy + scaleBias.zw;

	return vout;
}

//_________________________________________________________________________________________________

half4 blur1_px( VS_OUT1 In ) : COLOR 
{
	return float4(tex2D(diffuse_tex, In.offsetWeights[0].xy).rgb * tint.rgb * In.offsetWeights[0].z, 1);
}

//_________________________________________________________________________________________________

VS_OUT2 blur2_vx( VS_IN_2D input)
{
	VS_OUT2 vout;
	vout.position = input.position;
#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif
	
	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 2; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
	return vout;
}

//_________________________________________________________________________________________________

half4 blur2_px( VS_OUT2 In ) : COLOR
{
	half4 vColor = 0.5 * (tex2D(diffuse_tex, In.offsetWeights[0].xy) + tex2D(diffuse_tex, In.offsetWeights[1].xy));
    return float4(vColor.rgb * tint.rgb, 1);
}

//_________________________________________________________________________________________________

VS_OUT3 blur3_vx( VS_IN_2D input)
{
	VS_OUT3 vout;
	vout.position = input.position;
#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif

	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 3; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
	return vout;
}

//_________________________________________________________________________________________________

half4 blur3_px( VS_OUT3 In ) : COLOR
{
	half4 vColor = In.offsetWeights[0].z * tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += In.offsetWeights[1].z * tex2D(diffuse_tex, In.offsetWeights[1].xy);
	vColor += In.offsetWeights[2].z * tex2D(diffuse_tex, In.offsetWeights[2].xy);
    return float4(vColor.rgb * tint.rgb, 1);
}

//_________________________________________________________________________________________________

VS_OUT4 blur4_vx( VS_IN_2D input)
{
	VS_OUT4 vout;
	vout.position = input.position;
#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif

	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 4; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
	return vout;
}

//_________________________________________________________________________________________________

half4 blur4_px( VS_OUT4 In ) : COLOR
{
	half4 vColor = tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += tex2D(diffuse_tex, In.offsetWeights[1].xy);
	vColor += tex2D(diffuse_tex, In.offsetWeights[2].xy);
	vColor += tex2D(diffuse_tex, In.offsetWeights[3].xy);
	vColor *= 0.25;

    return float4(vColor.rgb * tint.rgb, 1);
}

//_________________________________________________________________________________________________

VS_OUT5 blur5_vx( VS_IN_2D input)
{
	VS_OUT5 vout;
	vout.position = input.position;
#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif

	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 5; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
	return vout;
}

//_________________________________________________________________________________________________

half4 blur5_px( VS_OUT5 In ) : COLOR
{
	half4 vColor = In.offsetWeights[0].z * tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += In.offsetWeights[1].z * tex2D(diffuse_tex, In.offsetWeights[1].xy);
	vColor += In.offsetWeights[2].z * tex2D(diffuse_tex, In.offsetWeights[2].xy);
	vColor += In.offsetWeights[3].z * tex2D(diffuse_tex, In.offsetWeights[3].xy);
	vColor += In.offsetWeights[4].z * tex2D(diffuse_tex, In.offsetWeights[4].xy);
    return float4(vColor.rgb * tint.rgb, 1);
}

//_________________________________________________________________________________________________

half4 blur5DoF_px( VS_OUT5 In ) : COLOR
{
	// Optimized version of the above with fewer weight multiplies and no tinting.
	half4 vColor = In.offsetWeights[0].z * tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += In.offsetWeights[1].z * (tex2D(diffuse_tex, In.offsetWeights[1].xy) + tex2D(diffuse_tex, In.offsetWeights[3].xy));
	vColor += In.offsetWeights[2].z * (tex2D(diffuse_tex, In.offsetWeights[2].xy) + tex2D(diffuse_tex, In.offsetWeights[4].xy));
    return vColor;
}

//_________________________________________________________________________________________________


half4 blur5DoFHQ_px( VS_OUT5 In ) : COLOR
{
	float4 texel, texel0 = tex2D(diffuse_tex, In.offsetWeights[0].xy);
	float maskWeight, weightSum = In.offsetWeights[0].z * texel0.a;
	float3 colour = weightSum * texel0.rgb;

	texel = tex2D(diffuse_tex, In.offsetWeights[1].xy);
	maskWeight = In.offsetWeights[1].z * texel.a;
	colour += maskWeight * texel.rgb;
	weightSum += maskWeight;

	texel = tex2D(diffuse_tex, In.offsetWeights[2].xy);
	maskWeight = In.offsetWeights[2].z * texel.a;
	colour += maskWeight * texel.rgb;
	weightSum += maskWeight;

	texel = tex2D(diffuse_tex, In.offsetWeights[3].xy);
	maskWeight = In.offsetWeights[3].z * texel.a;
	colour += maskWeight * texel.rgb;
	weightSum += maskWeight;

	texel = tex2D(diffuse_tex, In.offsetWeights[4].xy);
	maskWeight = In.offsetWeights[4].z * texel.a;
	colour += maskWeight * texel.rgb;
	weightSum += maskWeight;

	colour /= weightSum;
	return half4(colour, texel0.a);
}

//_________________________________________________________________________________________________


/**
 * Vertex shader for 7 tap blur
 */
VS_OUT7 blur7_vx( VS_IN_2D input)
{
	VS_OUT7 vout;
	vout.position = input.position;

#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif
	
	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 7; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;
		// scale and bias texture coordinates
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
	return vout;
}

//_________________________________________________________________________________________________

/**
 * Fragment shader for 7 tap blur
 */
half4 blur7_px( VS_OUT7 In ) : COLOR
{
	half4 vColor = In.offsetWeights[0].z * tex2D(diffuse_tex, In.offsetWeights[0].xy);
	vColor += In.offsetWeights[1].z * tex2D(diffuse_tex, In.offsetWeights[1].xy);
	vColor += In.offsetWeights[2].z * tex2D(diffuse_tex, In.offsetWeights[2].xy);
	vColor += In.offsetWeights[3].z * tex2D(diffuse_tex, In.offsetWeights[3].xy);
	vColor += In.offsetWeights[4].z * tex2D(diffuse_tex, In.offsetWeights[4].xy);
	vColor += In.offsetWeights[5].z * tex2D(diffuse_tex, In.offsetWeights[5].xy);
	vColor += In.offsetWeights[6].z * tex2D(diffuse_tex, In.offsetWeights[6].xy);
    return vColor;
}

//_________________________________________________________________________________________________

/**
 * Vertex shader for 9 tap blur
 */
VS_OUT9 blur9_vx( VS_IN_2D input)
{
	VS_OUT9 vout;
	vout.position = input.position;

#ifdef _WINPC
	half2 texcoord = input.uvSet0;
#else
	half2 texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif

	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	for (int i = 0; i < 9; i++)
	{
		vout.offsetWeights[i] = offsetWeights[i];
		vout.offsetWeights[i].xy += texcoord;
		// scale and bias texture coordinates
		vout.offsetWeights[i].xy = vout.offsetWeights[i].xy * scaleBias.xy + scaleBias.zw;
	}
	return vout;
}

//_________________________________________________________________________________________________

/**
 * Fragment shader for 9 tap blur
 */
half4 blur9_px( VS_OUT9 In ) : COLOR
{
	half4 vColor = In.offsetWeights[4].z * tex2D(diffuse_tex, In.offsetWeights[4].xy);
	vColor += In.offsetWeights[3].z * (tex2D(diffuse_tex, In.offsetWeights[3].xy) + tex2D(diffuse_tex, In.offsetWeights[5].xy));
	vColor += In.offsetWeights[2].z * (tex2D(diffuse_tex, In.offsetWeights[2].xy) + tex2D(diffuse_tex, In.offsetWeights[6].xy));
	vColor += In.offsetWeights[1].z * (tex2D(diffuse_tex, In.offsetWeights[1].xy) + tex2D(diffuse_tex, In.offsetWeights[7].xy));
	vColor += In.offsetWeights[0].z * (tex2D(diffuse_tex, In.offsetWeights[0].xy) + tex2D(diffuse_tex, In.offsetWeights[8].xy));
    return vColor;
}

//_________________________________________________________________________________________________

uniform float4 guardScaleBias : register(c128);

struct VS_OUTBLURGUARD
{
	float4 position : POSITION;
	half2 texcoord : TEXCOORD0;
};

//_________________________________________________________________________________________________

VS_OUTBLURGUARD blurguard_vx(VS_IN_2D input)
{
	VS_OUTBLURGUARD vout;
	
	vout.position = input.position;

#ifdef _WINPC
	vout.texcoord = input.uvSet0;
#else
	vout.texcoord = input.position.xy * half2(0.5, -0.5) + 0.5;
#endif

	vout.texcoord = vout.texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	vout.texcoord = vout.texcoord * guardScaleBias.xy + guardScaleBias.zw;
		
	return vout;
}

//_________________________________________________________________________________________________

// It is actually copytex_px
half4 blurguard_px( VS_OUTBLURGUARD In ) : COLOR
{
	return float4(0,0,0,0);//tex2D(diffuse_tex, In.texcoord);
}

//_________________________________________________________________________________________________
// 4-tap SM1 blur

#ifdef _WINPC

sampler2D color_tex0		: register(s0);
sampler2D color_tex1		: register(s1);
sampler2D color_tex2		: register(s2);
sampler2D color_tex3		: register(s3);

float2 SM1_UVOffsets[4]		: register(c0);
float4 SM1_ScaleBias		: register(c4);
float4 SM1_Weights			: register(c0);

struct VS_OUT_SM1
{
	float4 position		: POSITION;
	float2 texcoord0	: TEXCOORD0;
	float2 texcoord1	: TEXCOORD1;
	float2 texcoord2	: TEXCOORD2;
	float2 texcoord3	: TEXCOORD3;
};

VS_OUT_SM1 blur_SM1_vx( VS_IN_2D input)
{
	VS_OUT_SM1 OUT;
	
	OUT.position = input.position;
	float2 texcoord = input.uvSet0;
	
	texcoord = texcoord * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;

	OUT.texcoord0 = texcoord + SM1_UVOffsets[0].xy;
	OUT.texcoord0 = OUT.texcoord0 * SM1_ScaleBias.xy + SM1_ScaleBias.zw;

	OUT.texcoord1 = texcoord + SM1_UVOffsets[1].xy;
	OUT.texcoord1 = OUT.texcoord1 * SM1_ScaleBias.xy + SM1_ScaleBias.zw;

	OUT.texcoord2 = texcoord + SM1_UVOffsets[2].xy;
	OUT.texcoord2 = OUT.texcoord2 * SM1_ScaleBias.xy + SM1_ScaleBias.zw;

	OUT.texcoord3 = texcoord + SM1_UVOffsets[3].xy;
	OUT.texcoord3 = OUT.texcoord3 * SM1_ScaleBias.xy + SM1_ScaleBias.zw;
		
	return OUT;
}

float4 blur_SM1_px( VS_OUT_SM1 IN ) : COLOR
{
	float4 outColor = 0;
	
	outColor += tex2D(color_tex0, IN.texcoord0) * SM1_Weights.x;
	outColor += tex2D(color_tex1, IN.texcoord1) * SM1_Weights.y;
	outColor += tex2D(color_tex2, IN.texcoord2) * SM1_Weights.z;
	outColor += tex2D(color_tex3, IN.texcoord3) * SM1_Weights.w;
	
    return outColor;
}

#endif // _WINPC
